# ########################################################################
# Log Processing Script für FTP-Server zu Discord-Integration via Webhooks
# Autor: FMJ von "Die Alten Säcke" // scumsaecke.de // scumsäcke.de
# Mit freundlicher Unterstützung von Scumworld.de
# ########################################################################
# Version: 2.105 vom 15.02.2025
# ########################################################################
#
# Dieses Skript ist Open Source und darf frei verwendet, verändert und weitergegeben werden,
# solange es **nicht für kommerzielle Zwecke** genutzt wird. Der Verkauf oder das Anbieten gegen Bezahlung ist untersagt.
#
# Autor: FMJ (Phil) von scumsaecke.de
# Lizenz: CC-BY-NC 4.0 (https://creativecommons.org/licenses/by-nc/4.0/)
#
# ########################################################################
# ======================================================================
#   Skript-Beschreibung:
# ======================================================================
#
# Dieses Skript dient zur automatischen Überwachung von Spielserver-Logs 
# mit erweiterten Funktionen für SCUM-Server. Es kombiniert FTP-Logverarbeitung,
# Battlemetrics-Integration, Discord-Benachrichtigungen und automatische Wartung.

# ----------------------------------------------------------------------
#   Erweiterte Funktionsübersicht:
# ----------------------------------------------------------------------

# 1. Automatisierte Log-Verarbeitung:
#    - Download von Logs via FTP mit MLSD/NLST-Support
#    - Intelligente Filterung mittels filter.txt
#    - UTC-Zeitstempel-Konvertierung mit pytz
#    - SHA-256 Hashgenerierung für Einträge um ungewollte Doppelposts zu vermeiden

# 2. Discord-Integration:
#    - Multi-Webhook-Support für verschiedene Log-Kategorien
#    - Dynamische Nachrichtensegmentierung (<1750 Zeichen)
#    - Rate-Limit-Management mit konfigurierbaren Delays
#    - Automatisierte Start-/Neustart-Benachrichtigungen

# 3. Battlemetrics-Integration:
#    - Echtzeit-Serverstatistiken (Spielerzahlen, Status, Version)
#    - Dynamische Zeitanzeige im Discord-Format (<t:unix_ts>)

# 4. Wartungsfunktionen:
#    - Automatische Log-Archivierung (Jahr/Monat/Tag-Struktur)
#    - Retention Policy (Tage konfigurierbar)
#    - Selbstheilungsmechanismen:
#       • Auto-Update-Erkennung
#       • Modul-Selbstinstallation
#       • Konfigurationsfile-Reparatur

# 5. Erweiterte Überwachung:
#    - Echtzeit-FTP-Verbindungsdiagnose
#    - Datenbank-Integritätschecks
#    - Detaillierte Laufzeitstatistiken
#    - Rotierende Logfiles mit UTF-16LE-Encoding

# 6. Windows Tray-Icon Management:
#    - Tray Icon mit eigenem Icon-Bild
#    - Rechtsklick-Menü mit diversen dynamischen Submenüs
#    - Konsole aus- und einblenden (Background-Mode)

# ----------------------------------------------------------------------
#   Technische Highlights:
# ----------------------------------------------------------------------
# • Multi-Threading-fähige Architektur
# • Background-Running (Konsole ausblenden, Steuerung über Tray-Icon)
# • SQLite3-Datenbank mit Hash-basiertem Duplikatschutz
# • Adaptive Zeitzonenbehandlung (UTC/Lokal)
# • Konfigurierbare Retention Policies
# • Automatische Fehlerkorrektur bei:
#   - Fehlenden Konfigurationsdateien
#   - Ungültigen Zeitstempeln
#   - API-Ausfällen
#   - Encoding-Fehlern

# ----------------------------------------------------------------------
#   Einsatzgebiet:
# ----------------------------------------------------------------------
# Entwickelt für SCUM-Server-Administration mit Fokus auf:
# • Automatisierte Log-Verarbeitung inkl. Posting
# • Spielereignis-Monitoring
# • Cross-Platform-Benachrichtigungssystem (Discord)
# • Langzeit-Statistikerfassung // Wird modular hinzugefügt

# ======================================================================
# Solltest DU Fehler finden oder Verbesserungsvorschläge haben,
# kontaktiere uns im Scumworld Forum oder per Discord.
# ########################################################################

import configparser
import ftplib
import sqlite3
import hashlib
import os
import sys
import subprocess
import logging
import shutil
from datetime import datetime, timedelta
import re
import time
from PIL import Image, ImageDraw
from pystray import Icon, MenuItem, Menu
import threading
import ctypes

# externe Module für die automatische Installation, falls nicht vorhanden
required_modules = [
    'requests',
    'pytz',
    'pystray',
    'pillow'
]

import pytz
import requests

# Base-Config laden // Variablen setzen
# filestamp nur im .py Format (Entwicklermode)
file_path = globals().get("__file__", "")
if file_path and os.path.exists(file_path) and file_path.endswith(".py"):
    file_timestamp = os.path.getmtime(file_path)

script_check_count = 0

# ########################################################################
#    Log-Handling
# ########################################################################

# Log-Verzeichnis erstellen
log_dir = "logs"
os.makedirs(log_dir, exist_ok=True)

# Zeitstempel für den Log-Dateinamen
log_timestamp = datetime.now().strftime("%Y%m%d%H%M%S")
log_filename = os.path.join(log_dir, f"botlog_{log_timestamp}.log")

# Eigene Formatter-Klasse, um das Datumsformat ohne Millisekunden zu setzen
class CustomFormatter(logging.Formatter):
    def formatTime(self, record, datefmt=None):
        dt = datetime.fromtimestamp(record.created)
        return dt.strftime("%Y.%m.%d-%H.%M.%S:")  # Format ohne Millisekunden

# Logger erstellen
logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)

# Formatter mit dem angepassten Datumsformat
formatter = CustomFormatter("%(asctime)s - %(levelname)s - %(message)s")

# FileHandler für Datei-Logging
file_handler = logging.FileHandler(log_filename, mode="w", encoding="utf-16-le")
file_handler.setFormatter(formatter)

# StreamHandler für die Konsole
console_handler = logging.StreamHandler()
console_handler.setFormatter(formatter)

# Handler final zum Logger hinzufügen
logger.addHandler(file_handler)
logger.addHandler(console_handler)

def load_config(config_file='config.ini', filter_file='filter.txt', db_file='logs.db'):
# ########################################################################
#    Lädt die Konfigurationsdatei und setzt Standardwerte.
#    Prüft die Existenz wichtiger Dateien (filter.txt, logs.db, config.ini)
#    und erstellt sie, falls diese fehlen.
#    Gibt zurück: ConfigParser-Objekt
# ########################################################################
    config = configparser.ConfigParser()

    # Standardwerte für alle Abschnitte
    defaults = {
        'General': {
            'codename': 'LogBoB by FMJ',
            'show_intro': '1',
            'discord_message_limit': '1750',
            'log_dir': 'logs',
            'db_name': 'logs.db',
            'filter_file': 'filter.txt',
            'retention_hours': '24',
            'autoreboot': '0',
            'server_id': '12345678',
            'wait_time': '35',
            'updated': '0',
            'hoster': '1',
            'rate_limit_delay': '1',
            'archive_dir': 'Archiv',
            'archive': '1'
        },
        'FTP': {
            'host': 'deine FTP Server IP',
            'user': 'dein FTP User',
            'password': 'dein FTP Passwort',
            'remote_dir': '/remote_logordner',
            'port': 'dein Port'
        },
        'Database': {
            'db_path': 'logs.db',
            'days_to_keep': '2'
        },
        'Discord': {
            'admin': 'nicht_eingetragen',
            'chat': 'nicht_eingetragen',
            'chest_ownership': 'nicht_eingetragen',
            'economy': 'nicht_eingetragen',
            'event_kill': 'nicht_eingetragen',
            'famepoints': 'nicht_eingetragen',
            'gameplay': 'nicht_eingetragen',
            'kill': 'nicht_eingetragen',
            'login': 'nicht_eingetragen',
            'loot': 'nicht_eingetragen',
            'raid_protection': 'nicht_eingetragen',
            'vehicle_destruction': 'nicht_eingetragen',
            'violations': 'nicht_eingetragen',
            'botlog': 'nicht_eingetragen'
        }
    }

    # Strukturprüfung der notwendigen Dateien
    if not os.path.exists(filter_file):
        logger.warning(f"Es wurde keine Filterdatei gefunden, ich erstelle die Datei mit empfohlenen Werten.")
        with open(filter_file, 'w', encoding='utf-8') as f:
            # Füge die Inhalte in die Datei ein
            f.write("# Kommentare mit # am Zeilenanfang\n")
            f.write("Game version\n")
            f.write("Log: Parsing\n")
            f.write("ItemLootTreeNodes\n")
            f.write("disabled for spawning\n")
            f.write("0.000000\n")
            f.write("Bunker is Locked\n")
            f.write("Locked bunkers:\n")
            f.write("Error: 'RIS\n")
            f.write("Log:   0\n")
            f.write("Log:   1\n")
            f.write("Bunker activations:\n")
            f.write("Log: Filtered and sorted\n")
            f.write("------------------------------------------------------------\n")
            f.write("Sekunden...\n")
            f.write("LOG-Download:\n")
            f.write("Status: online\n")
            f.write("INFO - Zeit:\n")
            f.write("INFO - Version:\n")
            f.write("> Script ist aktuell!\n")
            f.write("INFO - Starte konfigurierte\n")
            f.write("INFO - Wartezeit abgelaufen\n")
            f.write("INFO - FTP-Verbindung erfolgreich\n")
            f.write("INFO - Datenbank wurde initialisiert\n")
            f.write("INFO - Verbinde mit FTP-Server\n")
            f.write("INFO - Datenbankwartung ist nicht notwendig\n")
            f.write("INFO - Starte Verarbeitungszyklus\n")
            f.write("INFO - Spieler:\n")
            f.write("INFO - --> Durchlauf\n")
            f.write("INFO - So! Auf gehts\n")
            f.write("INFO - Name:\n")
            f.write("Lokal gelöscht: 0 Dateien\n")
            f.write("Prüfe im\n")
            f.write("- Nachricht gesendet\n")
            f.write("- gesendet\n")
            f.write("- WARNING - ----------------------------------------------------------\n")
            f.write("Success: No. Elapsed time\n")
            f.write("FS-Wartung: Es wurden keine\n")
            f.write("Gelöschte Datenbank-Einträge:\n")
            f.write("\n# Ab hier Server spezifische Einträge hinzufügen\n")
            f.write("Pingperfect\n")

    
    if not os.path.exists(db_file):
        logger.warning(f"{db_file} nicht gefunden, erstelle die Datei.")
        # Hier könntest du die DB initialisieren, falls erforderlich
        open(db_file, 'w').close()  # Leere Datei erstellen

    try:
        config.read_file(open(config_file))
    except FileNotFoundError:
        # Datei existiert nicht, erstelle sie mit Standardwerten
        logger.warning(f"{config_file} nicht gefunden, erstelle die Datei mit Standardwerten.")
        config.read_dict(defaults)
        with open(config_file, 'w') as configfile:
            config.write(configfile)
        logger.info("Konfigurationsdatei wurde erstellt, bitte editieren.")
        # Benutzer zur Bestätigung auffordern, bevor das Skript beendet wird
        input("Das Skript wurde gestoppt. Bitte die config.ini bearbeiten und das Programm neu starten.\nDrücke Enter, um das Skript zu beenden.")
        sys.exit()
    else:
        # Eventuell Fehlende Abschnitte/Schlüssel ergänzen
        for section, values in defaults.items():
            if not config.has_section(section):
                config.add_section(section)
            for key, val in values.items():
                if not config.has_option(section, key):
                    config.set(section, key, val)

    return config

def check_required_modules(config):
# ########################################################################
#    Überprüft, ob alle externen Python-Module, die für das Skript benötigt werden, installiert sind.
#    Falls Module fehlen, wird der Benutzer gefragt, ob diese automatisch installiert werden sollen.
#    Bei erfolgreicher Installation wird das Skript neu gestartet.
#
#    Parameter:
#        config (configparser.ConfigParser): Die geladene Konfigurationsdatei.
#
#    Rückgabewert:
#        bool: True, wenn alle Module vorhanden sind oder die Prüfung übersprungen wird.
#              False, wenn der Benutzer die Installation ablehnt.
# ########################################################################

    # Nur prüfen, wenn updated=0
    if config['General'].getint('updated') == 1:
        return True

    # Liste der externen Module (nicht in der Standardlib)
    external_modules = ['requests', 'pytz']
    missing = []

    # Prüfen, welche Module fehlen
    for module in external_modules:
        try:
            __import__(module)
        except ImportError:
            missing.append(module)
    # Logik, wenn Module fehlen
    if missing:
        logger.error("FEHLER: Folgende Module sind nicht installiert:")
        logger.error(" -> %s", ", ".join(missing))

        # Benutzerabfrage zur Installation
        install = input("Möchtest du die fehlenden Module automatisch installieren? (J/N): ").strip().lower()
        if install == 'j':
            try:
                # Module installieren
                for module in missing:
                    logger.info(f"Installiere {module}...")
                    subprocess.run([sys.executable, "-m", "pip", "install", module], check=True)
                logger.info("Alle Module erfolgreich installiert.")

                # Konfiguration aktualisieren und Skript neu starten
                config.set('General', 'updated', '1')
                with open('config.ini', 'w') as configfile:
                    config.write(configfile)
                logger.info("Setze updated=1 in config.ini. Starte Skript neu...")

                # Skript neu starten
                python = sys.executable
                os.execv(python, [python] + sys.argv)
            except subprocess.CalledProcessError as e:
                logger.error(f"Fehler bei der Installation: {e}")
                return False
        else:
            logger.critical("Installation abgebrochen. Das Skript wird beendet.")
            return False
    else:
        # Alle Module vorhanden, updated=1 in config.ini setzen
        config.set('General', 'updated', '1')
        with open('config.ini', 'w') as configfile:
            config.write(configfile)
        logger.info("Alle externen Module vorhanden. Setze updated=1.")
        return True


def show_introduction():
# ########################################################################
#    Zeigt eine Einleitung in der Konsole an.
#    Diese Funktion wird einmalig beim Skriptstart aufgerufen.
# ########################################################################
    logger.info("-------------------------------------------------------------")
    time.sleep(0.5)
    logger.info("Log Processing Script für FTP-Server zu Discord-Integration via Webhooks")
    time.sleep(0.5)
    logger.info("By FMJ // scumsaecke.de // scumsäcke.de // Lizenz: CC-BY-NC 4.0")
    time.sleep(0.5)
    logger.info("Mit Unterstützung von Scumworld.de // ChatGPT // DeepSeek")
    time.sleep(0.5)
    logger.info("Das Programm wird initialisiert - bitte warten...")
    time.sleep(0.5)
    logger.info("-------------------------------------------------------------")
    time.sleep(2)
    logger.info("So! Auf gehts - es gibt viel zu tun...")
    time.sleep(1)


# ########################################################################
#    Tray-Handling
# ########################################################################
# Ctypes Initialisierung
ctypes.windll.kernel32.GetConsoleWindow.restype = ctypes.c_void_p

# Ressourcen-Handling für PyInstaller
def get_resource_path(relative_path):
    """Bekomme absoluten Pfad zur Ressource, funktioniert für dev und PyInstaller"""
    if hasattr(sys, '_MEIPASS'):
        return os.path.join(sys._MEIPASS, relative_path)
    return os.path.join(os.path.abspath("."), relative_path)

# Konsolenfenster-Steuerung
def get_console_window():
    return ctypes.windll.kernel32.GetConsoleWindow()

def is_console_available():
    return get_console_window() != 0

def is_console_visible():
    if not is_console_available():
        return False
    return ctypes.windll.user32.IsWindowVisible(get_console_window()) != 0

def show_console(icon, item):
    if is_console_available():
        ctypes.windll.user32.ShowWindow(get_console_window(), 1)  # SW_SHOWNORMAL
    update_menu(icon)

def hide_console(icon, item):
    if is_console_available():
        ctypes.windll.user32.ShowWindow(get_console_window(), 0)  # SW_HIDE
    update_menu(icon)

# Skript neu starten
def restart_script(icon, item):
    icon.stop()
    python = sys.executable
    os.execl(python, python, *sys.argv)

# Skript beenden
def exit_script(icon, item):
    icon.stop()
    if sys.platform == "win32" and is_console_available():
        ctypes.windll.kernel32.FreeConsole()
    sys.exit(0)

# Icon-Handling
def create_icon():
    try:
        return Image.open(get_resource_path("icon.ico"))
    except Exception as e:
        print(f"Fehler beim Laden des Icons: {e}")
        # Fallback weißes Quadrat
        return Image.new('RGB', (64, 64), (255, 255, 255))

# Menü-Handling
header_item = MenuItem(
    "=== LogBoB Menü ===", 
    lambda icon, item: None,
    enabled=False
)

def update_menu(icon):
    menu_items = [header_item]
    
    if is_console_available():
        if is_console_visible():
            menu_items.append(MenuItem("❌ Konsole ausblenden", hide_console))
        else:
            menu_items.append(MenuItem("🔳 Konsole anzeigen", show_console))
    
    menu_items.extend([
        MenuItem("🔄 Neu starten", restart_script),
        MenuItem("🚪 Beenden", exit_script)
    ])
    
    icon.menu = Menu(*menu_items)
    icon.update_menu()

def run_tray_icon():
    icon = Icon(
        name="LogBoB by FMJ",
        icon=create_icon(),
        title="LogBoB - Server Log Manager"
    )
    update_menu(icon)
    icon.run()


def ftp_connect(host, user, password, port):
# ########################################################################
#    Stellt eine FTP-Verbindung her
#    Gibt zurück: FTP-Objekt oder None bei Fehlern
# ########################################################################
    logger.info(f"Verbinde mit FTP-Server {host}:{port}...")
    try:
        ftp = ftplib.FTP()
        ftp.connect(host, int(port), timeout=15)
        ftp.login(user, password)
        logger.info("-------------------------------------------------------------")
        logger.info("FTP-Verbindung erfolgreich hergestellt - starte Log-Download")
        logger.info("-------------------------------------------------------------")
        return ftp
    except Exception as e:
        logger.error(f"FTP-Verbindungsfehler: {str(e)}")
        return None


def download_newest_log_files(ftp, remote_dir, config):
# ########################################################################
#    Lädt die neuesten Logdateien vom FTP-Server
#    Gibt zurück: Dictionary mit {log_type: local_path}
# ########################################################################
    if not ftp:
        return {}

    log_dir = config['General']['log_dir']
    os.makedirs(log_dir, exist_ok=True)

    try:
        ftp.cwd(remote_dir)
        
        # Konfigurationswert für passenden Hoster auslesen (Standard: 0 für nlst - zb für PingPerfect)
        use_mlsd = int(config['General'].get('hoster', 0))
        
        if use_mlsd:
            filenames = [entry[0] for entry in ftp.mlsd()]
        else:
            filenames = ftp.nlst()
        
    except Exception as e:
        logger.error(f"FTP-Verzeichniszugriff fehlgeschlagen: {str(e)}")
    
        if "502" in str(e):
            logger.error(f"Ein falscher Hoster wurde eingetragen! Bitte editiere den Hoster der config.ini")
            input("Das Skript wurde gestoppt. Bitte die config.ini bearbeiten und das Programm neu starten.\nDrücke Enter, um das Skript zu beenden.")
            sys.exit(0) 
        return {}

    # Dateien nach Typ und Zeitstempel filtern
    log_pattern = re.compile(r'^(?P<type>\w+)_(?P<timestamp>\d{14})\.log$')
    newest_files = {}

    for filename in filenames:
        if re.search(r'[\\/]', filename):  # Verzeichnisse ignorieren
            continue
        if match := log_pattern.match(filename):
            log_type = match.group('type')
            timestamp = match.group('timestamp')
            # Behalte nur die neueste Datei pro Typ
            if (log_type not in newest_files or 
                timestamp > newest_files[log_type][0]):
                newest_files[log_type] = (timestamp, filename)

    # Dateien herunterladen
    downloaded = {}
    for log_type, (_, filename) in newest_files.items():
        local_path = os.path.join(log_dir, filename)
        try:
            with open(local_path, 'wb') as f:
                ftp.retrbinary(f'RETR {filename}', f.write)
            logger.info(f"LOG-Download:   {filename}")
            downloaded[log_type] = local_path
        except Exception as e:
            logger.error(f"Download fehlgeschlagen: {filename} - {str(e)}")

    logger.info("-------------------------------------------------------------")
    return downloaded


def load_filter_patterns(config):
# ########################################################################
#    Lädt Filterbegriffe aus der Filterdatei
#    Gibt zurück: Liste der Filterpatterns
# ########################################################################
    filter_file = config['General']['filter_file']
    
    if not os.path.exists(filter_file):
        logger.warning(f"{filter_file} nicht gefunden, erstelle die Datei.")
    with open(filter_file, 'r', encoding='utf-8') as f:
        pass  # leere Filter-Datei leer
    
    patterns = []
    
    try:
        with open(filter_file, 'r', encoding='utf-8') as f:
            for line in f:
                cleaned = line.strip()
                if cleaned and not cleaned.startswith('#'):
                    patterns.append(cleaned)
    except FileNotFoundError:
        logger.warning(f"Filterdatei '{filter_file}' nicht gefunden")
    except Exception as e:
        logger.error(f"Fehler beim Lesen der Filterdatei: {str(e)}")
    
    return patterns


def parse_log_file(file_path, log_type, config):
# ########################################################################
#    Verarbeitet eine Logdatei zu strukturierten Einträgen
#    Gibt zurück: Liste von Logeinträgen
# ########################################################################
    entries = []
    current_block = []
    current_timestamp = None
    timestamp_pattern = re.compile(r'^(\d{4}\.\d{2}\.\d{2}-\d{2}\.\d{2}\.\d{2}): (.*)')
    separator_pattern = re.compile(r'-{30,}')

    try:
        with open(file_path, 'r', encoding='utf-16-le', errors='ignore') as f:
            for line in f:
                line = line.strip()
                if not line:
                    continue

                # Logblock-Erkennung
                if timestamp_pattern.match(line):
                    if current_timestamp:
                        process_block(current_block, current_timestamp, entries, file_path, log_type, config)
                    current_timestamp = line
                    current_block = [line]
                elif separator_pattern.fullmatch(line):
                    if current_timestamp:
                        process_block(current_block, current_timestamp, entries, file_path, log_type, config)
                    current_timestamp = None
                    current_block = []
                else:
                    current_block.append(line)

            # Restlichen Block verarbeiten
            if current_timestamp:
                process_block(current_block, current_timestamp, entries, file_path, log_type, config)

    except Exception as e:
        logger.error(f"Fehler beim Lesen von {file_path}: {str(e)}")
    
    return entries


def process_block(block_lines, timestamp_line, entries, file_path, log_type, config):
# ########################################################################
#    Verarbeitet einen einzelnen Logblock
#    Gibt zurück: Aktualisierte Eintragsliste inkl angepasster Timestamps
# ########################################################################
    timestamp_pattern = re.compile(r'^(\d{4}\.\d{2}\.\d{2}-\d{2}\.\d{2}\.\d{2}): (.*)')
    
    if not (match := timestamp_pattern.match(timestamp_line)):
        return entries

    timestamp_str = match.group(1)
    content_lines = [match.group(2)] + block_lines[1:]

    try:
        # Zeitstempel-Verarbeitung
        dt = datetime.strptime(timestamp_str, "%Y.%m.%d-%H.%M.%S")
        
        # UTC-Konvertierung nur bei Nicht-Botlog-Einträgen
        if log_type == 'botlog':
            dt_utc = dt  # Originalzeit ohne Zeitzonen-Anpassung verwenden
        else:
            dt_utc = pytz.utc.localize(dt)

            
        adjusted_time = dt_utc.strftime("%Y.%m.%d-%H.%M.%S")
    except ValueError as e:
        logger.error(f"Ungültiger Zeitstempel '{timestamp_str}': {str(e)}")
        return entries

    # Filterung der Einträge
    filtered = []
    patterns = load_filter_patterns(config)
    for line in content_lines:
        if not any(p in line for p in patterns):
            filtered.append(line)

    content = "\n".join(filtered).strip()
    if not content:  # Leere Blöcke ignorieren
        return entries

    # Eindeutigen Hash generieren
    unique_id = f"{os.path.basename(file_path)}-{timestamp_line}"
    entry_hash = hashlib.sha256(unique_id.encode('utf-16-le')).hexdigest()

    entries.append({
        'hash': entry_hash,
        'timestamp': adjusted_time,
        'log_type': log_type,
        'content': content
    })
    
    return entries


def init_db(config):
# ########################################################################
#    Initialisiert die SQLite-Datenbank
#    Gibt zurück: Datenbankverbindung
# ########################################################################
    db_path = config['Database'].get('db_path', config['General']['db_name'])
    conn = sqlite3.connect(db_path)
    
    with conn:
        conn.execute('''
            CREATE TABLE IF NOT EXISTS logs (
                hash TEXT PRIMARY KEY,
                timestamp TEXT NOT NULL,
                log_type TEXT NOT NULL,
                content TEXT NOT NULL,
                sent INTEGER DEFAULT 0,
                sent_filtered INTEGER DEFAULT 0
            )
        ''')
    
    logger.info(f"Datenbank wurde initialisiert: {db_path}")
    return conn


def insert_log_entry(conn, entry):
# ########################################################################
#    Fügt einen Logeintrag in die Datenbank ein
#    Gibt zurück: True bei Erfolg, False bei Fehlern
# ########################################################################
    try:
        with conn:
            conn.execute(
                """INSERT OR IGNORE INTO logs 
                (hash, timestamp, log_type, content)
                VALUES (?, ?, ?, ?)""",
                (entry['hash'], entry['timestamp'], entry['log_type'], entry['content'])
            )
        return True
    except sqlite3.Error as e:
        logger.error(f"Datenbankfehler: {str(e)}")
        return False


def cleanup_db(conn, config):
# ########################################################################
#    Löscht alte Einträge aus der Datenbank
# ########################################################################
    days = int(config['Database']['days_to_keep'])
    cutoff = datetime.now() - timedelta(days=days)
    cutoff_str = cutoff.strftime("%Y.%m.%d-%H.%M.%S")
    
    
    with conn:
        deleted = conn.execute(
            "DELETE FROM logs WHERE timestamp < ?",
            (cutoff_str,)
        ).rowcount
    
    if deleted:
        logger.info("-------------------------------------------------------------")
        logger.info(f"DB-Wartung: Gelöschte Datenbank-Einträge: {deleted}")
    else:
        logger.info("-------------------------------------------------------------")
        logger.info(f"Datenbankwartung ist nicht notwendig - wird übersprungen.")


def send_discord_message(webhook_url, message):
    # ########################################################################
    #    Sendet eine Nachricht an Discord
    #    Gibt zurück: True bei Erfolg, False bei Fehlern
    # ########################################################################
    if not message.strip():
        return True  # Falls die Nachricht leer ist, kein Senden notwendig

    try:
        response = requests.post(
            webhook_url,
            json={"content": message},
            timeout=15
        )
        response.raise_for_status()  # Löst Fehler für 4xx und 5xx aus
        return True
    except requests.exceptions.HTTPError as http_err:
        if response.status_code == 401:
            logger.error(f"Discord-Sendefehler: 401 Unauthorized - Webhook möglicherweise ungültig.")
        elif response.status_code == 403:
            logger.error(f"Discord-Sendefehler: 403 Forbidden - Fehlende Berechtigung oder Rate Limit erreicht.")
        elif response.status_code == 404:
            logger.error(f"Discord-Sendefehler: 404 Not Found - Webhook existiert nicht.")
        elif response.status_code == 429:
            logger.error(f"Discord-Sendefehler: 429 Too Many Requests - Rate Limit überschritten.")
        else:
            logger.error(f"Discord-Sendefehler: HTTP {response.status_code} - {response.text}")
    except requests.exceptions.Timeout:
        logger.error("Discord-Sendefehler: Anfrage hat das Zeitlimit überschritten.")
    except requests.exceptions.RequestException as req_err:
        logger.error(f"Discord-Sendefehler: {str(req_err)}")
    
    return False  # Falls ein Fehler auftritt, False zurückgeben


def send_unsent_logs(conn, log_type, webhook_url, config):
# ########################################################################
#    Verarbeitet und sendet ungelesene Logs
# ########################################################################
    message_limit = int(config['General']['discord_message_limit'])
    # Wartezeit zwischen den Nachrichten um Rate-Limit an der DC API zu vermeiden
    rate_limit_delay = float(config['General'].get('rate_limit_delay', 1))
    cursor = conn.cursor()

    with conn:
        cursor.execute(
            """SELECT hash, timestamp, content 
            FROM logs 
            WHERE sent = 0 AND log_type = ? 
            ORDER BY timestamp ASC""",
            (log_type,)
        )
        entries = cursor.fetchall()

    current_message = ""
    hashes_to_mark = []

    for entry_hash, timestamp_str, content in entries:
        # Zeitstempel-Verarbeitung
        try:
            # Parse den Zeitstempel aus dem Log-Eintrag
            dt = datetime.strptime(timestamp_str, "%Y.%m.%d-%H.%M.%S")
            
            # UTC-Konvertierung nur bei Nicht-Botlog-Einträgen
            if log_type != 'botlog':
                dt = pytz.utc.localize(dt)  # Zeitzonen-Anpassung für Nicht-Botlog-Einträge
            else:
                dt = dt  # Originalzeit ohne Zeitzonen-Anpassung für Botlog-Einträge
            
            # Unix-Zeitstempel berechnen
            unix_ts = int(dt.timestamp())  # timestamp() berücksichtigt bereits die Zeitzone
            
            # Discord-Nachricht formatieren
            line = f"**[**<t:{unix_ts}:T>**]** `{content}`\n"
            
        except Exception as e:
            logger.error(f"Zeitstempel-Fehler: {str(e)}")
            continue

        # Nachricht aufteilen bei Überschreitung des Zeichen-Limits
        if len(current_message) + len(line) > message_limit:
            if send_discord_message(webhook_url, current_message):
                with conn:
                    conn.executemany(
                        "UPDATE logs SET sent = 1 WHERE hash = ?",
                        [(h,) for h in hashes_to_mark]
                    )
            current_message = line
            hashes_to_mark = [entry_hash]
            time.sleep(rate_limit_delay)
        else:
            current_message += line
            hashes_to_mark.append(entry_hash)

    # Restliche Nachricht senden
    if current_message:
        if send_discord_message(webhook_url, current_message):
            logger.info(f"---> Logpost: {unix_ts} - {log_type} - gesendet")
            with conn:
                conn.executemany(
                    "UPDATE logs SET sent = 1 WHERE hash = ?",
                    [(h,) for h in hashes_to_mark]
                )
            time.sleep(rate_limit_delay)


def archive_local_logs(config):
# ########################################################################
# Archiviert lokale Log-Dateien, die älter als 24 Stunden sind
# ########################################################################
    log_dir = config['General']['log_dir']
    archive_dir = config['General']['archive_dir']
    max_age = timedelta(hours=24)
    now = datetime.now()
    
    try:
        for filename in os.listdir(log_dir):
            if filename.endswith('.log'):
                file_path = os.path.join(log_dir, filename)
                file_mtime = datetime.fromtimestamp(os.path.getmtime(file_path))
                if (now - file_mtime) > max_age:
                    # Erstelle die Ordnerstruktur für das Archiv: Jahr/Monat/Tag
                    year_folder = os.path.join(archive_dir, str(file_mtime.year))
                    month_folder = os.path.join(year_folder, file_mtime.strftime('%m'))
                    day_folder = os.path.join(month_folder, file_mtime.strftime('%d')) 
                    
                    # Archiv-Ordner erstellen, falls sie noch nicht existieren
                    os.makedirs(day_folder, exist_ok=True)
                    
                    # Datei in den entsprechenden Archiv-Ordner verschieben
                    archive_path = os.path.join(day_folder, filename)
                    shutil.move(file_path, archive_path)
                    
                    logger.info(f"Archiviert: {filename} nach {archive_path}")
    
    except Exception as e:
        logger.error(f"FS-Wartung: Archivierung fehlgeschlagen: {str(e)}")

def cleanup_local_logs(config):
# ########################################################################
# Löscht lokale Log-Dateien älter als 24 Stunden
# ########################################################################
    log_dir = config['General']['log_dir']
    max_age = timedelta(hours=24)
    now = datetime.now()
    deleted_count = 0
    
    try:
        for filename in os.listdir(log_dir):
            if filename.endswith('.log'):
                file_path = os.path.join(log_dir, filename)
                file_mtime = datetime.fromtimestamp(os.path.getmtime(file_path))
                if (now - file_mtime) > max_age:
                    os.remove(file_path)
                    deleted_count += 1
        if deleted_count == 0:
            logger.info(f"FS-Wartung: Es wurden keine lokalen Dateien gelöscht")  
        else:
            logger.info(f"FS-Wartung: Lokal gelöscht: {deleted_count} Dateien")

    except Exception as e:
        logger.error(f"FS-Wartung: Lokale Bereinigung fehlgeschlagen: {str(e)}")

def get_server_info(server_id):
# ########################################################################
#    Fragt über die Battlemetrics-API die aktuellen Serverinformationen ab.
#    Es werden u.a. die Spielerzahl, die Spielversion, der Serverstatus 
#    und (falls vorhanden) die Serverzeit zurückgegeben.
# ########################################################################
    url = f"https://api.battlemetrics.com/servers/{server_id}"
    try:
        response = requests.get(url)
        response.raise_for_status()  # löst bei HTTP-Fehlern eine Exception aus
    except requests.RequestException as e:
        logging.error(f"Fehler bei der API-Anfrage: {e}")
        return None

    try:
        data = response.json()
        attributes = data["data"]["attributes"]
        # Extrahieren der benötigten Server-Informationen:
        server_info = {
            "id": attributes.get("id"),
            "name": attributes.get("name"),
            "ip": attributes.get("ip"),
            "port": attributes.get("port"),
            "players": attributes.get("players"),
            "maxPlayers": attributes.get("maxPlayers"),
            "status": attributes.get("status"),
            "version": attributes.get("details", {}).get("version", "Unbekannt"),
            "serverTime": attributes.get("details", {}).get("time", "Unbekannt")
        }
        return server_info
    except (ValueError, KeyError) as e:
        logging.error(f"Fehler beim Verarbeiten der API-Antwort: {e}")
        return None

def countdown(total_wait):
# ########################################################################
# Gibt einen Countdown in der Konsole mit Zwischenmeldungen aus.
# ########################################################################
    logger.info("-------------------------------------------------------------")
    logger.info(f"Starte konfigurierte Wartezeit von {total_wait} Sekunden!")
    spinner = ['/','-','\\','|']  # Die Zeichen für den Spinner
    spinner_index = 0  # Index, um das aktuelle Zeichen zu wählen
    
    for remaining in range(total_wait, 0, -1):
        time.sleep(1)
        
        # Bei bestimmten Zeiten (z.B. jede Minute) eine Nachricht ausgeben
        if remaining in {300,240,120,90,60,30,15,10,5}:
            logger.info(f"Noch {remaining} Sekunden...")
        
        # Spinner-Zeichen ausgeben und überschreiben
        sys.stdout.write(f"\r Ich warte... c[_]   {spinner[spinner_index]}\r")  
        sys.stdout.flush()
        
        # Spinner-Zeichen wechseln
        spinner_index = (spinner_index + 1) % len(spinner)
    
    logger.info(f"Wartezeit abgelaufen - Starte neuen Durchlauf!")

def check_file_timestamp():
# ########################################################################
# Prüft, ob das Skript geändert wurde, und startet es ggf. neu.
# ########################################################################
    if __file__.endswith(".py"):
        return
    global file_timestamp
    new_timestamp = os.path.getmtime(__file__)
    if new_timestamp != file_timestamp:
        logger.warning("----------------------------------------------------------")
        logger.warning(">>> Code wurde aktualisiert. Neustart erforderlich! <<<<< ")
        logger.warning("----------------------------------------------------------")
        file_timestamp = new_timestamp
        restart_script()
    else:
        global script_check_count
        script_check_count += 1
        logger.info(f'> Script ist aktuell! {file_timestamp} ({script_check_count}. Prüfung)')

def restart_script():
# ########################################################################
# Startet das Skript bei Änderung automtisch neu.
# ########################################################################
    logger.info("Skript wird neu gestartet...")
    python = sys.executable
    os.execv(python, [python] + sys.argv)

def send_welcome_message(webhook_url, config):
# ########################################################################
# Sendet eine Startnachricht mit Versionsnummer und Startzeit an Discord
# ########################################################################
    if not webhook_url:
        return

    # Bot-Namen aus der config.ini lesen, Standardwert "FMJs Logscript", falls nicht vorhanden
    bot_name = config['General'].get('codename', 'FMJs Logscript')

    now = datetime.now(pytz.UTC)  
    unix_ts = int(now.timestamp())

    # Verwende den Bot-Namen aus der Konfiguration
    start_message = f"**[**<t:{unix_ts}:T>**]** `- >>>> - {bot_name} ({unix_ts}) wurde gestartet!`"

    try:
        response = requests.post(webhook_url, json={"content": start_message}, timeout=15)
        if response.status_code != 204:
            logger.error(f"Fehler beim Senden der Nachricht: {response.status_code} - {response.text}")
    except Exception as e:
        logger.error(f"Fehler beim Senden der Willkommensnachricht an Discord: {str(e)}")

#####################################################################
########################## MAIN - FUNKTION ##########################
#####################################################################

def main(required_modules):
    loopcounter = 0
    config = load_config()
    total_wait = int(config['General'].get('wait_time', 30))
    autoreboot = int(config['General'].get('autoreboot', 0))
    
    if "__file__" in globals() and __file__.endswith(".py"):
        config = load_config() 
        if not check_required_modules(config):
            sys.exit(1)  # Beenden, wenn Module fehlen
    
    # Webhooks für Info-Kanäle aus der Config laden
    welcome_webhook = config['Discord'].get('botlog', '').strip()
    
    # Discord-Zeichenlimit aus der Config laden (Standard: 1750 Zeichen)
    message_limit = int(config['General'].get('discord_message_limit', 1750))
    
    # Willkommensnachricht senden, falls konfiguriert
    send_welcome_message(welcome_webhook, config)
    
    # FTP-Verbindungsparameter aus der Config laden
    required_ftp = ['host', 'user', 'password', 'remote_dir']
    if any(key not in config['FTP'] for key in required_ftp):
        logger.critical("Ungültige FTP-Konfiguration")
        return
    ftp_host = config['FTP']['host']
    ftp_user = config['FTP']['user']
    ftp_pass = config['FTP']['password']
    ftp_dir = config['FTP']['remote_dir']
    ftp_port = config['FTP'].get('port', 21)  # Standard-Port, falls nicht gesetzt
    
    # Discord-Log Webhooks auslesen (z.B. für verschiedene Log-Typen)
    discord_webhooks = dict(config['Discord'].items()) if 'Discord' in config else {}
    
    # Battlemetrics Server-ID aus [General] auslesen
    server_id = config['General'].get('server_id', None)
    if not server_id:
        logger.warning("Keine Battlemetrics server_id in der config.ini gefunden. Battlemetrics-Abfrage wird übersprungen.")
    
    while True:
# ##########################################
# ####### H A U P T S C H L E I F E ########
# ##########################################
        config = load_config()
        if "__file__" in globals() and __file__.endswith(".py") and autoreboot:
            check_file_timestamp()

        loopcounter += 1  # Durchlaufzähler erhöhen
        logger.info("-------------------------------------------------------------")
        logger.info(f"--> Durchlauf {loopcounter} gestartet...")
        logger.info("-------------------------------------------------------------")

        # Battlemetrics-Abfrage, falls server_id vorhanden
        info = get_server_info(server_id)
        if info:
            time.sleep(0.2)
            logger.info(f"{info['name']}")
            time.sleep(0.2)
            logger.info(f"Spieler: {info['players']} von {info['maxPlayers']}")
            time.sleep(0.2)
            logger.info(f"Status: {info['status']}")
            time.sleep(0.2)
            logger.info(f"Version: {info['version']}")
            time.sleep(0.2)
            logger.info(f"Zeit: {info['serverTime']}")
            time.sleep(0.2)
            logger.info(f"IP: {info['ip']}:{info['port']}")
            time.sleep(0.2)
            logger.info("-------------------------------------------------------------")
        else:
            logger.warning("Konnte Serverinformationen nicht abrufen.")
            logger.info("-------------------------------------------------------------")
        logger.info("Starte Verarbeitungszyklus...")
        
        conn = None
        ftp = None 
        try:
            # Datenbankverbindung herstellen
            conn = init_db(config)

            # FTP-Verbindung herstellen
            ftp = ftp_connect(ftp_host, ftp_user, ftp_pass, ftp_port)
            if ftp:
                # Heruntergeladene Dateien verarbeiten
                downloaded = download_newest_log_files(ftp, ftp_dir, config)
                for log_type, file_path in downloaded.items():
                    entries = parse_log_file(file_path, log_type, config)
                    for entry in entries:
                        insert_log_entry(conn, entry)

                discord_webhook_regex = r"^https://discord(app)?\.com/api/webhooks/\d+/\S+$"
                
                # Discord-Nachrichten senden (nur wenn gültige Webhook-URL)
                for log_type, webhook in discord_webhooks.items():
                    if re.match(discord_webhook_regex, webhook):  # Nur gültige URLs verwenden
                        send_unsent_logs(conn, log_type, webhook, config)
                    else:
                        print(f"Information: Ich erkenne eine ungültige Webhook-URL für {log_type}")

                # Datenbankbereinigung
                cleanup_db(conn, config)

            # Lokale Log-Bereinigung (Dateien älter als 24 Stunden)
            archive_status = config['General']['archive']
            if archive_status == '1':
                logger.info(f"FS-Wartung: Prüfe im Archiv-Modus")
                archive_local_logs(config)
            else: 
                logger.info(f"FS-Wartung: Archivierung deaktiviert - Prüfe im Cleanup-Modus")
                cleanup_local_logs(config)

            # Verarbeitung der lokalen botlog-Dateien
            for filename in os.listdir(log_dir):
                if filename.startswith("botlog_") and filename.endswith(".log"):
                    file_path = os.path.join(log_dir, filename)
                    # Verwende "botlog" als log_type für Discord Channel Zuweisung
                    entries = parse_log_file(file_path, "botlog", config)
                    for entry in entries:
                        insert_log_entry(conn, entry)

        except KeyboardInterrupt:
            logger.info("Skript wurde manuell beendet")
            break
        except Exception as e:
            logger.error(f"Kritischer Fehler: {str(e)}", exc_info=True)
        finally:
            if conn:
                conn.close()
            if ftp:
                try:
                    # Prüfen, ob die FTP-Verbindung noch aktiv ist
                    if hasattr(ftp, 'sock') and ftp.sock is not None:
                        ftp.quit()
                    else:
                        logger.warning("FTP-Verbindung bereits geschlossen oder ungültig")
                except Exception as e:
                    logger.error(f"Fehler beim Schließen der FTP-Verbindung: {e}")
        # Countdown bis zum nächsten Durchlauf
        countdown(total_wait)


if __name__ == "__main__":
    # Tray-Icon in einem separaten Thread starten
    tray_thread = threading.Thread(target=run_tray_icon, daemon=True)
    tray_thread.start()

    # Nachdem das Tray-Icon läuft, führe die restlichen Funktionen aus
    config = load_config()
    show_intro_enabled = int(config['General'].get('show_intro', 1))
    if show_intro_enabled == 1:
        show_introduction()
    
    main(required_modules)