import tkinter as tk
from tkinter import ttk, simpledialog, messagebox
import configparser
from ttkbootstrap import Style
import os
import sys

class ConfigEditor:
    def __init__(self, master):
        self.master = master
        self.master.title("LogBoB Konfigurationsassistent")
        # Fenstergröße festlegen (Breite x Höhe)
        self.master.geometry("900x825")
        # ConfigParser initialisieren
        self.config = configparser.ConfigParser()
        
        # Config-Datei laden/erstellen
        self.config.read('config.ini')
        if not self.config.has_section('General'):
            self.config['General'] = {'theme': 'flatly'}
        if not self.config.has_section('Discord'):
            self.config['Discord'] = {}
        
        # Theme-System
        self.ui_themes = {'Dark': 'darkly', 'Bright': 'flatly'}
        self.current_theme = self.config.get('General', 'theme', fallback='flatly')
        self.style = Style(theme=self.current_theme)
        self.style.configure('Toggle.TButton', font=('Arial', 14))
        
        self.create_widgets()
        self.load_config()

    def create_widgets(self):
        self.notebook = ttk.Notebook(self.master)
        self.tabs = {
            'Einstellungen': ttk.Frame(self.notebook),  # Kombinierter Tab
            'Discord': ttk.Frame(self.notebook)         # Discord bleibt separat
        }
    
        for name, frame in self.tabs.items():
            self.notebook.add(frame, text=name)
        
        self.create_settings_tab()  # Neuer kombinierter Tab
        self.create_discord_tab()   # Discord-Tab bleibt unverändert
        
        self.notebook.pack(expand=1, fill='both', padx=10, pady=10)

        # Footer-Bereich
        btn_frame = ttk.Frame(self.master)
        
        # Theme-Toggle
        self.theme_slider = ttk.Button(
            btn_frame,
            text="☀️", 
            command=self.toggle_theme,
            style='Toggle.TButton',
            width=3
        )
        self.theme_slider.pack(side=tk.LEFT, padx=5)
        self.update_slider_emoji()

        # Aktionsbuttons
        ttk.Button(btn_frame, text="Speichern", command=self.save_config, style='success.TButton').pack(side=tk.RIGHT, padx=5)
        ttk.Button(btn_frame, text="Abbrechen", command=self.safe_exit, style='danger.TButton').pack(side=tk.RIGHT)
        btn_frame.pack(fill=tk.X, padx=10, pady=10, anchor='s')

    def create_settings_tab(self):
        tab = self.tabs['Einstellungen']
        
        # 1. Allgemeine Einstellungen
        ttk.Label(tab, text="Allgemeine Einstellungen", font=('Helvetica', 10, 'bold')).pack(pady=(10, 5), anchor='w')
        ttk.Separator(tab).pack(fill=tk.X, pady=5)
        
        general_fields = [
            ('codename', 'Codename', 'Anwendungs-Codename (z.B. "LogBoB by FMJ")'),
            ('show_intro', 'Intro anzeigen', '1 = Intro zeigen, 0 = deaktivieren'),
            ('log_dir', 'Log-Verzeichnis', 'Ordner für Logdateien (z.B. "logs")'),
            ('retention_hours', 'Aufbewahrungszeit', 'Stunden bis Archivierung (empf. 24)'),
        ]
        for option, label, expl in general_fields:
            self.create_field(tab, 'General', option, label, expl)
        
        # 2. FTP-Einstellungen
        ttk.Label(tab, text="FTP-Einstellungen", font=('Helvetica', 10, 'bold')).pack(pady=(20, 5), anchor='w')
        ttk.Separator(tab).pack(fill=tk.X, pady=5)
        
        ftp_fields = [
            ('host', 'Server-IP', 'IP-Adresse des FTP-Servers'),
            ('port', 'Port', 'FTP-Port (z.B. 6657)'),
            ('user', 'Benutzername', 'FTP-Loginname'),
            ('password', 'Passwort', 'FTP-Passwort'),
            ('remote_dir', 'Remote-Ordner', 'Zielverzeichnis auf dem Server'),
        ]
        for option, label, expl in ftp_fields:
            self.create_field(tab, 'FTP', option, label, expl)
        
        # 3. Datenbank-Einstellungen
        ttk.Label(tab, text="Datenbank-Einstellungen", font=('Helvetica', 10, 'bold')).pack(pady=(20, 5), anchor='w')
        ttk.Separator(tab).pack(fill=tk.X, pady=5)
        
        db_fields = [
            ('days_to_keep', 'Aufbewahrungsdauer', 'Tage bis Bereinigung (empf. 2)'),
        ]
        for option, label, expl in db_fields:
            self.create_field(tab, 'Database', option, label, expl)

    def create_discord_tab(self):
        # Alte Einträge löschen
        for widget in self.tabs['Discord'].winfo_children():
            widget.destroy()
        
        self.discord_entries = []
        frame = ttk.Frame(self.tabs['Discord'])
        frame.pack(fill=tk.BOTH, expand=True)
        
        # Überschrift
        ttk.Label(frame, text="Discord-Webhooks", font=('Helvetica', 10, 'bold')).pack(pady=(10, 5), anchor='w')
        ttk.Separator(frame).pack(fill=tk.X, pady=5)
        
        # Hinweis-Text
        ttk.Label(
            frame, 
            text="Hier kannst du Discord-Webhooks für verschiedene Log-Typen konfigurieren.\n"
                "Nicht eingetragene Hooks werden vom System nicht gepostet, aber trotzdem verarbeitet.",
            foreground="#666666",
            wraplength=600
        ).pack(pady=(0, 10), anchor='w')
        
        # Vorhandene Webhooks
        for option in [k for k in self.config['Discord'] if self.config['Discord'][k].strip()]:
            self.add_discord_entry(option)
        

    def create_field(self, parent, section, option, label, explanation):
        frame = ttk.Frame(parent)
        frame.pack(fill=tk.X, padx=5, pady=8)
        
        ttk.Label(frame, text=label, width=20).pack(side=tk.LEFT)
        entry = ttk.Entry(frame)
        entry.pack(side=tk.LEFT, fill=tk.X, expand=True)
        entry.section = section
        entry.option = option
        
        ttk.Label(frame, text=explanation, foreground="#666666", wraplength=600).pack(side=tk.LEFT, padx=10)
        return entry

    def add_discord_entry(self, option):
        # Frame für den Webhook-Eintrag erstellen
        frame = ttk.Frame(self.tabs['Discord'])
        frame.pack(fill=tk.X, padx=5, pady=5)
        
        # Label für den Webhook-Namen
        ttk.Label(frame, text=f"{option}", width=20).pack(side=tk.LEFT)
        
        # Eingabefeld für die Webhook-URL
        entry = ttk.Entry(frame)
        entry.pack(side=tk.LEFT, fill=tk.X, expand=True)
        
        # Vorhandene URL laden (falls vorhanden)
        entry.insert(0, self.config['Discord'].get(option, ''))
        
        # Metadaten für das Speichern zuweisen
        entry.section = 'Discord'
        entry.option = option
        
        # Erklärungstext für das Eingabefeld
        ttk.Label(
            frame, 
            text="Voller Discord Webhook-Link\n(z.B. https://discord.com/api/webhooks/...)",
            foreground="#666666",
            wraplength=400
        ).pack(side=tk.LEFT, padx=10)
        
        # Löschen-Button
        ttk.Button(
            frame,
            text="X",
            command=lambda: self.remove_discord_entry(frame, option),
            style='danger.TButton',
            width=3
        ).pack(side=tk.RIGHT)
        
        # Entry zur Liste hinzufügen
        self.discord_entries.append(entry)

    def remove_discord_entry(self, frame, option):
        frame.destroy()
        del self.config['Discord'][option]
        self.discord_entries = [e for e in self.discord_entries if e.option != option]

    def toggle_theme(self):
        new_theme = 'darkly' if self.current_theme == 'flatly' else 'flatly'
        self.style.theme_use(new_theme)
        self.current_theme = new_theme
        self.config['General']['theme'] = new_theme
        self.update_slider_emoji()

    def update_slider_emoji(self):
        self.theme_slider['text'] = "🌙" if self.current_theme == 'darkly' else "☀️"

    def load_config(self):
        # Allgemeine Einstellungen
        for section in self.config.sections():
            for entry in self.get_all_entries():
                if entry.section == section and entry.option in self.config[section]:
                    entry.delete(0, tk.END)
                    entry.insert(0, self.config[section][entry.option])

    def get_all_entries(self):
        entries = []
        for tab in self.tabs.values():
            for child in tab.winfo_children():
                if isinstance(child, ttk.Frame):
                    for widget in child.winfo_children():
                        if isinstance(widget, ttk.Entry):
                            entries.append(widget)
        return entries

    def safe_exit(self):
        self.master.destroy()

    def save_config(self):
        # Aktualisiere alle Werte
        for entry in self.get_all_entries():
            if not self.config.has_section(entry.section):
                self.config.add_section(entry.section)
            self.config[entry.section][entry.option] = entry.get()
        
        # Leere Discord-Einträge entfernen
        self.config['Discord'] = {k:v for k,v in self.config['Discord'].items() if v.strip()}
        self.config_path = 'config.ini'
        with open(self.config_path, 'w') as f:
            self.config.write(f)
        
        messagebox.showinfo("Erfolg", "Konfiguration gespeichert!")

if __name__ == '__main__':
    root = tk.Tk()
    app = ConfigEditor(root)
    root.protocol("WM_DELETE_WINDOW", app.safe_exit)
    root.mainloop()