import configparser
import datetime
import requests
from dateutil.parser import parse
import os
import time
import sys

def global_exception_handler(exctype, value, traceback):
    """
    Globaler Exception Handler, der alle unbehandelten Exceptions abfängt
    und verhindert, dass sich die Konsole sofort schließt.
    """
    print(f"\nEin unerwarteter Fehler ist aufgetreten:")
    print(f"Typ: {exctype.__name__}")
    print(f"Details: {value}")
    input("\nDrücke Enter zum Beenden...")
    sys.exit(1)

# Registriere den globalen Exception Handler
sys.excepthook = global_exception_handler

# cfg_file: Name der Hauptkonfigurationsdatei
cfg_file = "ini\\_embed_poster_routing.ini"

def initialize_config(cfg_file):
    """
    Initialisiert die Routing-Konfigurationsdatei.

    Diese Funktion erstellt die Routing-Konfigurationsdatei `_embed_poster_routing.ini` mit
    drei Beispielroutings, falls sie nicht existiert. Die gerouteten INI-Dateien
    werden vom Webserver geladen.

    :param cfg_file: Pfad zur Routing-Konfigurationsdatei (_embed_poster_routing.ini)
    :return: Eine Liste von INI-Dateien und ein configparser.ConfigParser-Objekt
    """
    # Erstelle ein ConfigParser-Objekt
    config = configparser.ConfigParser(allow_no_value=True)

    # Extrahiere den Ordnerpfad aus dem cfg_file
    config_dir = os.path.dirname(cfg_file)
    
    # Wenn ein Ordnerpfad existiert und noch nicht vorhanden ist, erstelle ihn
    if config_dir and not os.path.exists(config_dir):
        try:
            os.makedirs(config_dir)
            print(f"Verzeichnis '{config_dir}' wurde erstellt.")
        except Exception as e:
            print(f"Fehler beim Erstellen des Verzeichnisses '{config_dir}': {e}")
            exit()

    # Prüfen, ob die Routing-Konfigurationsdatei existiert
    if not os.path.exists(cfg_file):
        # Wenn nicht, erstelle die Datei mit Beispielroutings direkt als Text
        config_text = """[Settings]
active_ini: embed_1.ini
;Weitere Embeds/Configs einfach durch entfernen des Semikolons hinzufügen

;active_ini_2: embed_2.ini
;active_ini_3: embed_3.ini

;Falls mehrere inis aktiv sind, wird die Zeit zwischen den Posts in Sekunden angegeben.
multipost_delay: 5
        """
        
        # Schreibe den Text direkt in die Datei
        with open(cfg_file, "w", encoding="utf-8") as f:
            f.write(config_text)
        print(f"Routing-Konfigurationsdatei '{cfg_file}' wurde erstellt.")

    # Lese die Routing-Konfigurationsdatei ein
    try:
        with open(cfg_file, "r", encoding="utf-8") as f:
            config.read_file(f)
    except Exception as e:
        print(f"Fehler beim Lesen der Routing-Konfigurationsdatei: {e}")
        input("Drücke Enter zum Beenden...")
        exit()

    # Sammle alle INI-Dateien aus den Schlüsseln, die mit "active_ini" beginnen
    active_inis = []
    for key, value in config.items("Settings"):
        if key.startswith("active_ini") and value.strip():
            ini_path = os.path.join("ini", value.strip())
            active_inis.append(ini_path)

    # Entferne doppelte Einträge
    active_inis = list(dict.fromkeys(active_inis))

    if active_inis:
        # Fehlende Dateien erstellen
        for ini_file in active_inis:
            if not os.path.exists(ini_file):
                print(f"INI-Datei '{ini_file}' existiert nicht. Erstelle sie...")
                create_default_ini(ini_file)
        return active_inis, config
    else:
        print("Keine Routing-INI-Dateien angegeben.")
        input("Drücke Enter zum Beenden...")
        exit()


def create_default_ini(ini_file):
    """
    Erstellt eine neue INI-Datei mit Standardwerten von einem Webserver.

    Diese Funktion lädt die Standard-Konfiguration von einem Webserver und
    erstellt damit eine neue INI-Datei. Falls der Download fehlschlägt,
    wird das Programm beendet.

    :param ini_file: Pfad zur zu erstellenden INI-Datei
    """
    config_url = "https://www.scumsaecke.de/res/bot/default_embed_poster_config.ini"
    
    try:
        # Konfiguration vom Webserver laden
        response = requests.get(config_url)
        response.raise_for_status()  # Wirft Exception bei HTTP-Fehlern
        
        # Kodierung explizit setzen, falls nicht automatisch erkannt
        response.encoding = 'utf-8'  # Erzwinge UTF-8-Kodierung
        default_config = response.text
        
        # Entferne nur führende und nachfolgende Leerzeichen, aber behalte leere Zeilen
        cleaned_config = "\n".join([line.rstrip() for line in default_config.splitlines()])
        
        # Konfiguration in Datei schreiben
        with open(ini_file, "w", encoding="utf-8") as f:
            f.write(cleaned_config)
            
        print(f"Standard-INI-Datei '{ini_file}' wurde erfolgreich vom Server geladen und erstellt.")
        
    except Exception as e:
        print(f"Fehler beim Laden der Konfiguration vom Server: {e}")
        input("Drücke Enter zum Beenden...")
        exit()


def replace_placeholders(text, days, date_str, time_str, datetime_str):
    """
    Ersetzt Platzhalter im Text durch dynamische Werte.

    Diese Funktion ersetzt die Platzhalter im Text durch die entsprechenden Werte:
    {days} - Anzahl der Tage seit Startdatum
    {date} - Aktuelles Datum
    {time} - Aktuelle Uhrzeit
    {datetime} - Aktuelles Datum und Uhrzeit
    
    :param text: Der Text mit Platzhaltern
    :param days: Anzahl der Tage seit Startdatum
    :param date_str: Aktuelles Datum formatiert
    :param time_str: Aktuelle Uhrzeit formatiert
    :param datetime_str: Aktuelles Datum und Uhrzeit formatiert
    :return: Text mit ersetzten Platzhaltern
    """
    return (
        text.replace("{days}", str(days))
        .replace("{date}", date_str)
        .replace("{time}", time_str)
        .replace("{datetime}", datetime_str)
        .replace(r'\n', '\n')
    )


def process_ini_files(ini_files, multipost_delay):
    """
    Verarbeitet eine Liste von INI-Dateien und sendet die entsprechenden Webhooks.
    """
    successful_sends = 0
    
    for i, ini_file in enumerate(ini_files):
        if not os.path.exists(ini_file):
            print(f"INI-Datei '{ini_file}' wurde nicht gefunden. Überspringe...")
            continue

        print(f"\nVerarbeite INI-Datei: {ini_file}")
        
        # Lese die aktuelle INI-Datei
        current_config = configparser.ConfigParser(allow_no_value=True)
        try:
            # Versuche zuerst die Datei zu lesen, um Syntaxfehler zu finden
            with open(ini_file, "r", encoding="utf-8") as f:
                content = f.read()
                
            try:
                # Versuche die Konfiguration zu parsen
                current_config.read_string(content)
            except configparser.Error as e:
                # Fange spezifische INI-Parser Fehler ab
                line_number = get_error_line(str(e))
                print(f"Fehler in der INI-Datei '{ini_file}' in Zeile {line_number}:")
                print_problematic_line(content, line_number)
                continue
                
        except Exception as e:
            print(f"Fehler beim Lesen der INI-Datei '{ini_file}': {e}")
            continue

        # Berechne die Tage seit dem Startdatum
        try:
            days = calculate_days(current_config.get("Settings", "start_date", fallback="2025-01-01"))
        except Exception as e:
            print(f"Fehler beim Berechnen der Tage für '{ini_file}': {e}")
            continue

        try:
            # Erstelle und sende den Embed
            embed = create_embed(days, current_config)
            if send_webhook(embed, current_config):
                successful_sends += 1
                # Verzögerung zwischen den Posts nur wenn es nicht der letzte erfolgreiche Post ist
                if multipost_delay > 0 and i < len(ini_files) - 1:
                    print(f"Warte {multipost_delay} Sekunden vor dem nächsten Post...")
                    time.sleep(multipost_delay)
            else:
                print(f"Fehler beim Senden des Webhooks für '{ini_file}'")
        except Exception as e:
            print(f"Fehler beim Erstellen/Senden des Embeds für '{ini_file}': {e}")
            continue

    print(f"\nVerarbeitung abgeschlossen. {successful_sends} von {len(ini_files)} Embeds erfolgreich gesendet.")
    if successful_sends == 0:
        input("Keine Embeds erfolgreich gesendet. Drücke Enter zum Beenden...")
        sys.exit(1)
    else:
        input("Drücke Enter zum Beenden...")
        sys.exit(0)


def get_error_line(error_message):
    """
    Extrahiert die Zeilennummer aus einer Fehlermeldung.
    """
    import re
    # Suche nach "line X" im Fehlertext
    match = re.search(r"line (\d+)", error_message)
    if match:
        return int(match.group(1))
    return None


def print_problematic_line(content, line_number):
    """
    Zeigt die problematische Zeile mit Kontext an.
    """
    if line_number is None:
        print("Konnte die fehlerhafte Zeile nicht lokalisieren.")
        return

    lines = content.splitlines()
    if 1 <= line_number <= len(lines):
        # Zeige bis zu 2 Zeilen vor und nach der problematischen Zeile
        start = max(0, line_number - 3)
        end = min(len(lines), line_number + 2)
        
        print("\nProblematischer Bereich:")
        for i in (start, end):
            prefix = ">>>" if i + 1 == line_number else "   "
            print(f"{prefix} Zeile {i+1}: {lines[i]}")
        print()


def calculate_days(start_date_str):
    """
    Berechnet die Anzahl der Tage seit einem gegebenen Startdatum.

    Diese Funktion nimmt ein Startdatum im Format `yyyy-mm-dd` entgegen
    und berechnet die Anzahl der Tage, die seit diesem Datum vergangen sind.

    :param start_date_str: Das Startdatum im Format `yyyy-mm-dd`
    :return: Die Anzahl der Tage seit dem Startdatum
    """
    try:
        start_date = parse(start_date_str).date()
        today = datetime.date.today()
        return (today - start_date).days
    except Exception as e:
        print(f"Fehler beim Parsen des Datums: {e}")
        input("Drücke Enter zum Beenden...")
        exit()


def create_embed(days_diff, current_config):
    """
    Erstellt ein Discord Embed-Objekt basierend auf der aktuellen Konfiguration.

    :param days_diff: Die Anzahl der Tage seit dem Startdatum
    :param current_config: Die aktuelle Konfiguration (configparser.ConfigParser-Objekt)
    :return: Ein Dictionary, das das Discord Embed-Objekt repräsentiert
    """
    now = datetime.datetime.now()
    date_format = current_config.get("Header", "date_format", fallback="%d.%m.%Y")
    time_format = current_config.get("Header", "time_format", fallback="%H:%M:%S")
    datetime_format = current_config.get("Header", "datetime_format", fallback="%d.%m.%Y %H:%M:%S")

    current_date = now.strftime(date_format)
    current_time = now.strftime(time_format)
    current_datetime = now.strftime(datetime_format)
    
    title = replace_placeholders(
        current_config.get("Header", "title", fallback=""),
        days_diff,
        current_date,
        current_time,
        current_datetime
    )
    description = replace_placeholders(
        current_config.get("Header", "description", fallback=""),
        days_diff,
        current_date,
        current_time,
        current_datetime
    )
    color = int(current_config.get("Style", "color", fallback="3498db"), 16)

    # Felder aus der Konfiguration laden und sortieren
    fields = []
    field_sections = [section for section in current_config.sections() if section.startswith("Field_")]
    field_sections.sort(key=lambda x: int(x.split("_")[1]))
    
    for section in field_sections:
        name = replace_placeholders(
            current_config.get(section, "Überschrift", fallback=""),
            days_diff,
            current_date,
            current_time,
            current_datetime
        )
        value = replace_placeholders(
            current_config.get(section, "Inhalt", fallback=""),
            days_diff,
            current_date,
            current_time,
            current_datetime
        )
        inline = current_config.getboolean(section, "inline", fallback=False)
        if name or value:
            fields.append({
                "name": name,
                "value": value,
                "inline": inline
            })
    
    footer_text = replace_placeholders(
        current_config.get("Footer", "text", fallback="Discord embed Poster made by FMJ - Scumsaecke.de\nGepostet am {datetime}"),
        days_diff,
        current_date,
        current_time,
        current_datetime
    )
    thumbnail_url = current_config.get("Thumbnail", "url", fallback=None)
    image_url = current_config.get("Image", "url", fallback=None)
    author_name = replace_placeholders(
        current_config.get("Author", "name", fallback=""),
        days_diff,
        current_date,
        current_time,
        current_datetime
    )
    author_icon = current_config.get("Author", "icon_url", fallback=None)
    
    embed = {
        "title": title,
        "description": description,
        "color": color,
        "fields": fields,
        "footer": {"text": footer_text}
    }

    if thumbnail_url:
        embed["thumbnail"] = {"url": thumbnail_url}
    
    if image_url:
        embed["image"] = {"url": image_url}
    
    if author_name or author_icon:
        embed["author"] = {
            "name": author_name,
            "icon_url": author_icon
        }

    return embed


def send_webhook(embed, config):
    """
    Sendet ein Discord Embed-Objekt über einen Webhook.
    """
    try:
        test_mode = config.getboolean("Test", "test-mode", fallback=True)
    except Exception as e:
        print(f"Fehler beim Lesen von 'test-mode': {e}")
        return False

    try:
        if test_mode:
            if not config.has_option("Test", "test_webhook_url"):
                print("Fehler: 'test_webhook_url' fehlt im Abschnitt [Test]")
                return False
            webhook_url = config.get("Test", "test_webhook_url")
        else:
            if not config.has_option("Settings", "webhook_url"):
                print("Fehler: 'webhook_url' fehlt im Abschnitt [Settings]")
                return False
            webhook_url = config.get("Settings", "webhook_url")
    
        payload = {
            "embeds": [embed],
            "username": config.get("Settings", "username", fallback="Daily Bot")
        }

        response = requests.post(webhook_url, json=payload)
        return handle_response(response)

    except requests.exceptions.RequestException as e:
        print(f"Fehler beim Senden des Webhooks: {e}")
        return False
        input("Drücke Enter zum Beenden...")

    except Exception as e:
        print(f"Unerwarteter Fehler: {e}")
        return False


def handle_response(response):
    """
    Verarbeitet die Antwort des Webhook-Servers.
    """
    status_messages = {
        200: "Erfolgreich gesendet!",
        201: "Erfolgreich gesendet!",
        204: "Erfolgreich gesendet!",
        400: "Ungültige Anfrage. Bitte überprüfe die Payload - Json-Formatierung in der ini. Prüfe deine Eingaben sorgfältig.",
        401: "Unerlaubter Zugriff. Bitte überprüfe die Webhook-URL.",
        403: "Zugriff verweigert. Bitte überprüfe die Berechtigungen des Webhooks.",
        404: "Der Webhook ist falsch oder veraltet. Bitte überprüfen und in der ini aktualisieren!",
        429: "Zu viele Anfragen. Bitte später erneut versuchen.",
        500: "Serverfehler. Bitte später erneut versuchen."
    }

    message = status_messages.get(response.status_code, f"Fehler: {response.status_code}, Antwort: {response.text}")
    print(message)

    return response.status_code in {200, 201, 204}


# Main-Loop
if __name__ == "__main__":
    ini_files, config = initialize_config(cfg_file)
    multipost_delay = config.getint("Settings", "multipost_delay", fallback=0)
    process_ini_files(ini_files, multipost_delay)