import os
import sys
import subprocess
import datetime
import tkinter as tk
from tkinter import messagebox, ttk
import psutil
import ctypes
from pathlib import Path

def is_admin():
    """Check if the script is running with admin privileges."""
    try:
        return ctypes.windll.shell32.IsUserAnAdmin()
    except:
        return False

def close_processes():
    """Close SCUM and Steam processes."""
    closed = []
    for proc in psutil.process_iter(['pid', 'name']):
        try:
            proc_name = proc.info['name'].lower()
            if 'scum' in proc_name or 'steam' in proc_name:
                proc.terminate()
                closed.append(proc_name)
        except (psutil.NoSuchProcess, psutil.AccessDenied, psutil.ZombieProcess):
            pass
    
    if closed:
        messagebox.showinfo("Prozesse beendet", f"Folgende Prozesse wurden beendet: {', '.join(closed)}")
    else:
        messagebox.showinfo("Information", "Keine SCUM oder Steam Prozesse gefunden.")
    
    return True

def reset_cache():
    """Rename the SCUM cache folder with timestamp."""
    appdata_local = os.path.join(os.environ['LOCALAPPDATA'], 'SCUM')
    
    if not os.path.exists(appdata_local):
        messagebox.showinfo("Information", "SCUM Cache-Ordner existiert nicht.")
        return True
    
    # Create backup with timestamp
    timestamp = datetime.datetime.now().strftime("%Y%m%d_%H%M%S")
    new_path = f"{appdata_local}_{timestamp}"
    
    try:
        os.rename(appdata_local, new_path)
        messagebox.showinfo("Cache zurückgesetzt", f"Der Cache wurde umbenannt in:\n{os.path.basename(new_path)}")
        return True
    except Exception as e:
        messagebox.showerror("Fehler", f"Cache konnte nicht zurückgesetzt werden:\n{str(e)}")
        return False

def reset_network():
    """Reset network connections using ipconfig /renew."""
    try:
        result = subprocess.run(
            ["ipconfig", "/renew"], 
            capture_output=True, 
            text=True, 
            creationflags=subprocess.CREATE_NO_WINDOW
        )
        
        if result.returncode == 0:
            messagebox.showinfo("Netzwerk zurückgesetzt", "Die Netzwerkverbindungen wurden erfolgreich zurückgesetzt.")
            return True
        else:
            messagebox.showerror("Fehler", f"Fehler beim Zurücksetzen des Netzwerks:\n{result.stderr}")
            return False
    except Exception as e:
        messagebox.showerror("Fehler", f"Fehler beim Zurücksetzen des Netzwerks:\n{str(e)}")
        return False

def execute_all():
    """Execute all troubleshooting steps in sequence."""
    steps_completed = 0
    total_steps = 3
    
    # Show progress dialog
    progress_window = tk.Toplevel()
    progress_window.title("Führe alle Schritte aus")
    progress_window.geometry("300x150")
    progress_window.resizable(False, False)
    progress_window.transient()
    progress_window.grab_set()
    progress_window.configure(bg="#1e1e1e")
    
    # Center the progress window
    progress_window.update_idletasks()
    width = progress_window.winfo_width()
    height = progress_window.winfo_height()
    x = (progress_window.winfo_screenwidth() // 2) - (width // 2)
    y = (progress_window.winfo_screenheight() // 2) - (height // 2)
    progress_window.geometry('{}x{}+{}+{}'.format(width, height, x, y))
    
    # Progress frame
    progress_frame = ttk.Frame(progress_window, padding=20, style="Dark.TFrame")
    progress_frame.pack(fill=tk.BOTH, expand=True)
    
    status_label = ttk.Label(progress_frame, text="Starte Troubleshooting...", style="Dark.TLabel")
    status_label.pack(pady=(0, 10))
    
    progress = ttk.Progressbar(progress_frame, orient="horizontal", length=250, mode="determinate", style="Dark.Horizontal.TProgressbar")
    progress.pack(pady=10)
    progress["maximum"] = total_steps
    
    # Update function to run steps in sequence
    def update_progress(step, message):
        status_label.config(text=message)
        progress["value"] = step
        progress_window.update()
    
    # Execute steps with progress updates
    update_progress(0, "Beende SCUM und Steam Prozesse...")
    progress_window.update()
    progress_window.after(500)  # Short delay for UI update
    if close_processes():
        steps_completed += 1
        update_progress(steps_completed, "Prozesse wurden beendet!")
    
    progress_window.after(1000)
    update_progress(steps_completed, "Setze SCUM Cache zurück...")
    if reset_cache():
        steps_completed += 1
        update_progress(steps_completed, "Cache wurde zurückgesetzt!")
    
    progress_window.after(1000)
    update_progress(steps_completed, "Setze Netzwerkverbindungen zurück...")
    if reset_network():
        steps_completed += 1
        update_progress(steps_completed, "Netzwerk wurde zurückgesetzt!")
    
    # Show final message
    progress_window.after(1000)
    if steps_completed == total_steps:
        status_label.config(text="Alle Schritte erfolgreich abgeschlossen!")
    else:
        status_label.config(text=f"{steps_completed} von {total_steps} Schritten abgeschlossen")
    
    # Close button
    close_btn = ttk.Button(progress_frame, text="Schließen", command=progress_window.destroy, style="Dark.TButton")
    close_btn.pack(pady=(10, 0))

def create_gui():
    """Create the main GUI window."""
    root = tk.Tk()
    root.title("SCUM Troubleshooting Tool by FMJ")
    root.geometry("450x780")
    root.resizable(False, False)
    
    # Set window icon if available
    try:
        root.iconbitmap("scum_icon.ico")
    except:
        pass
    
    # Configure dark modern style
    style = ttk.Style()
    style.theme_use("clam")  # Using clam as base theme
    
    # Configure dark colors
    bg_color = "#1e1e1e"  # Dunkler Hintergrund
    accent_color = "#3498db"  # Akzentfarbe (Blau)
    button_bg = "#3498db"  # Button Farbe
    button_hover = "#2980b9"  # Button hover Farbe
    section_bg = "#2d2d2d"  # Dunkler Abschnittsbereich
    text_color = "#f0f0f0"  # Heller Text
    execute_color = "#27ae60"  # Grün für Execute-Button
    
    # Frame Styles
    style.configure("TFrame", background=bg_color)
    style.configure("Card.TFrame", background=section_bg, relief="solid", borderwidth=1)
    
    # Label Styles
    style.configure("TLabel", 
                    background=bg_color, 
                    foreground=text_color, 
                    font=("Segoe UI", 10))
    
    style.configure("Header.TLabel", 
                    background=bg_color, 
                    foreground=text_color, 
                    font=("Segoe UI", 16, "bold"))
    
    style.configure("Title.TLabel", 
                    background=section_bg, 
                    foreground=text_color, 
                    font=("Segoe UI", 11, "bold"))
    
    style.configure("Footer.TLabel", 
                    background=bg_color, 
                    foreground=text_color, 
                    font=("Segoe UI", 8))
    
    # Text style für Beschreibungen (nicht bold)
    style.configure("Description.TLabel", 
                    background=section_bg, 
                    foreground=text_color, 
                    font=("Segoe UI", 10))
    
    # Dark styles for progress dialog
    style.configure("Dark.TFrame", background="#1e1e1e")
    style.configure("Dark.TLabel", background="#1e1e1e", foreground=text_color)
    style.configure("Dark.TButton", 
                    background=button_bg, 
                    foreground="white", 
                    font=("Segoe UI", 10),
                    padding=8)
    
    # Progress bar style
    style.configure("Dark.Horizontal.TProgressbar", 
                    background=accent_color,
                    troughcolor="#444444")
    
    # Button styles
    style.configure("TButton", 
                    background=button_bg, 
                    foreground="white", 
                    font=("Segoe UI", 10),
                    padding=8)
    
    style.configure("Action.TButton", 
                    background=button_bg, 
                    foreground="white",
                    font=("Segoe UI", 10),
                    padding=8)
    
    style.configure("Execute.TButton", 
                    background="#27ae60", 
                    foreground="white",
                    font=("Segoe UI", 11, "bold"),
                    padding=10)
    
    # Mouse hover effects for buttons (if your tkinter version supports it)
    try:
        style.map("TButton",
                background=[("active", button_hover)],
                foreground=[("active", "white")])
        style.map("Action.TButton",
                background=[("active", button_hover)],
                foreground=[("active", "white")])
        style.map("Execute.TButton",
                background=[("active", "#219952")],
                foreground=[("active", "white")])
        style.map("Dark.TButton",
                background=[("active", button_hover)],
                foreground=[("active", "white")])
    except:
        pass
    
    # Set window background
    root.configure(bg=bg_color)
    
    # Main container
    main_frame = ttk.Frame(root)
    main_frame.pack(fill=tk.BOTH, expand=True, padx=20, pady=20)
    
    # Header section
    header_label = ttk.Label(main_frame, text="SCUM Troubleshooting Tool by FMJ", style="Header.TLabel")
    header_label.pack(pady=(0, 15))
    
    desc_label = ttk.Label(main_frame, 
                           text="Mit diesem Tool können häufige Probleme in SCUM behoben werden.",
                           wraplength=400)
    desc_label.pack(pady=(0, 20))
    
    # Create card-like sections for each action
    # Action 1: Close processes
    action1_frame = ttk.Frame(main_frame, style="Card.TFrame")
    action1_frame.pack(fill=tk.X, pady=5, ipady=5)
    
    action1_title = ttk.Label(action1_frame, text="1. Prozesse beenden", style="Title.TLabel")
    action1_title.pack(anchor=tk.W, padx=10, pady=(5, 0))
    
    action1_desc = ttk.Label(action1_frame, 
                            text="Beendet alle laufenden SCUM und Steam Prozesse.",
                            wraplength=400,
                            style="Description.TLabel")
    action1_desc.pack(anchor=tk.W, padx=10, pady=(2, 5))
    
    close_btn = ttk.Button(
        action1_frame, 
        text="SCUM und Steam beenden", 
        command=close_processes,
        style="Action.TButton"
    )
    close_btn.pack(padx=10, pady=(0, 10), fill=tk.X)
    
    # Action 2: Reset cache
    action2_frame = ttk.Frame(main_frame, style="Card.TFrame")
    action2_frame.pack(fill=tk.X, pady=5, ipady=5)
    
    action2_title = ttk.Label(action2_frame, text="2. Cache zurücksetzen", style="Title.TLabel")
    action2_title.pack(anchor=tk.W, padx=10, pady=(5, 0))
    
    action2_desc = ttk.Label(action2_frame, 
                            text="Setzt den SCUM Cache zurück, indem der Ordner umbenannt wird.",
                            wraplength=400,
                            style="Description.TLabel")
    action2_desc.pack(anchor=tk.W, padx=10, pady=(2, 5))
    
    cache_btn = ttk.Button(
        action2_frame, 
        text="SCUM Cache zurücksetzen", 
        command=reset_cache,
        style="Action.TButton"
    )
    cache_btn.pack(padx=10, pady=(0, 10), fill=tk.X)
    
    # Action 3: Reset network
    action3_frame = ttk.Frame(main_frame, style="Card.TFrame")
    action3_frame.pack(fill=tk.X, pady=5, ipady=5)
    
    action3_title = ttk.Label(action3_frame, text="3. Netzwerk zurücksetzen", style="Title.TLabel")
    action3_title.pack(anchor=tk.W, padx=10, pady=(5, 0))
    
    action3_desc = ttk.Label(action3_frame, 
                            text="Setzt die Netzwerkverbindungen durch ipconfig /renew zurück.",
                            wraplength=400,
                            style="Description.TLabel")
    action3_desc.pack(anchor=tk.W, padx=10, pady=(2, 5))
    
    network_btn = ttk.Button(
        action3_frame, 
        text="Netzwerkverbindungen zurücksetzen", 
        command=reset_network,
        style="Action.TButton"
    )
    network_btn.pack(padx=10, pady=(0, 10), fill=tk.X)
    
    # Execute all button (highlighted) in a card frame for consistency
    execute_all_frame = ttk.Frame(main_frame, style="Card.TFrame")
    execute_all_frame.pack(fill=tk.X, pady=10, ipady=5)
    
    execute_all_title = ttk.Label(execute_all_frame, text="Alle Schritte ausführen", style="Title.TLabel")
    execute_all_title.pack(anchor=tk.W, padx=10, pady=(5, 0))
    
    execute_all_desc = ttk.Label(execute_all_frame, 
                            text="Führt alle oben genannten Schritte nacheinander aus.",
                            wraplength=400,
                            style="Description.TLabel")
    execute_all_desc.pack(anchor=tk.W, padx=10, pady=(2, 5))
    
    # Verwende einen normalen tk.Button statt ttk.Button für den Execute-Button
    # um sicherzustellen, dass die Farbe korrekt angezeigt wird
    execute_all_btn = tk.Button(
        execute_all_frame,
        text="✓ ALLE SCHRITTE AUSFÜHREN",
        command=execute_all,
        bg=execute_color,
        fg="white",
        font=("Segoe UI", 11, "bold"),
        relief=tk.FLAT,
        activebackground="#219952",
        activeforeground="white",
        padx=5,
        pady=8
    )
    execute_all_btn.pack(padx=10, pady=(0, 10), fill=tk.X)
    
    # Warning about admin rights for network reset
    if not is_admin():
        admin_frame = ttk.Frame(main_frame)
        admin_frame.pack(fill=tk.X, pady=(10, 0))
        
        admin_warning = ttk.Label(
            admin_frame, 
            text="⚠️ Zum Zurücksetzen der Netzwerkverbindungen werden\nAdministratorrechte benötigt.", 
            foreground="#e74c3c",
            justify="center"
        )
        admin_warning.pack()
    
    # Footer
    footer = ttk.Label(main_frame, text="© 2025 SCUM Troubleshooting Tool by FMJ - scumsaecke.de", style="Footer.TLabel")
    footer.pack(side=tk.BOTTOM, pady=(15, 0))
    
    # Center the window on screen
    root.update_idletasks()
    width = root.winfo_width()
    height = root.winfo_height()
    x = (root.winfo_screenwidth() // 2) - (width // 2)
    y = (root.winfo_screenheight() // 2) - (height // 2)
    root.geometry('{}x{}+{}+{}'.format(width, height, x, y))
    
    return root

if __name__ == "__main__":
    # Check for admin rights for network operations
    if not is_admin() and "renew" in " ".join(sys.argv).lower():
        ctypes.windll.shell32.ShellExecuteW(None, "runas", sys.executable, " ".join(sys.argv), None, 1)
        sys.exit(0)
    
    root = create_gui()
    root.mainloop()